// app.js
// ======================================================
// Transmisor WebRTC — código extraído y modularizado
// - Mantiene funcionalidad original
// - Añade selector de tema (oscuro/claro) con persistencia
// - Añade 3 ranuras de configuración (1–3) para guardar/cargar
// ======================================================

// ---------- Tema (oscuro / claro) ----------
const themeToggle = document.getElementById('themeToggle');
(function initTheme(){
  try{
    const saved = localStorage.getItem('ui_theme') || 'dark';
    document.documentElement.setAttribute('data-theme', saved);
    if(themeToggle) themeToggle.checked = (saved === 'dark');
  }catch{}
})();
themeToggle?.addEventListener('change', ()=>{
  const mode = themeToggle.checked ? 'dark' : 'light';
  document.documentElement.setAttribute('data-theme', mode);
  try{ localStorage.setItem('ui_theme', mode); }catch{}
});

// ---------- UI refs ----------
const micSelect = document.getElementById('mic');
const salaSelect = document.getElementById('sala');
const startBtn = document.getElementById('start');
const stopBtn = document.getElementById('stop');
const muteBtn = document.getElementById('muteBtn');
const clearLogBtn = document.getElementById('clearLog');
const usuariosConectados = document.getElementById('usuariosConectados');
const traficoDiv = document.getElementById('trafico');
const logArea = document.getElementById('log');
const toast = document.getElementById('toast');

const volumen = document.getElementById('volumen');
const volLabel = document.getElementById('volLabel');
const meterEl = document.getElementById('meter');
const peakHoldEl = document.getElementById('peakHold');
const rmsDbEl = document.getElementById('rmsDb');
const peakDbEl = document.getElementById('peakDb');
const clipDot = document.getElementById('clipDot');
const monitorChk = document.getElementById('monitor');

const agc = document.getElementById('agc');
const echo = document.getElementById('echo');
const noise = document.getElementById('noise');
const applyAudioBtn = document.getElementById('applyAudio');
const saveCfg = document.getElementById('saveCfg');
const loadCfg = document.getElementById('loadCfg');
const cfgSlot = document.getElementById('cfgSlot');

const cmpOn = document.getElementById('cmpOn');
const hpfOn = document.getElementById('hpfOn');
const lpfOn = document.getElementById('lpfOn');
const cmpThresh = document.getElementById('cmpThresh');
const cmpRatio  = document.getElementById('cmpRatio');
const cmpKnee   = document.getElementById('cmpKnee');
const cmpAttack = document.getElementById('cmpAttack');
const cmpRelease= document.getElementById('cmpRelease');
const hpfFreq   = document.getElementById('hpfFreq');
const lpfFreq   = document.getElementById('lpfFreq');
const applyFxBtn = document.getElementById('applyFx');
const resetFxBtn = document.getElementById('resetFx');

const presetDirecto = document.getElementById('presetDirecto');
const presetVoz     = document.getElementById('presetVoz');
const presetAgc     = document.getElementById('presetAgc');
const presetEntrevista = document.getElementById('presetEntrevista');
const presetRadio   = document.getElementById('presetRadio');
const presetReset   = document.getElementById('presetReset');

// ---------- Estado ----------
let ws, rawStream = null, localStream = null;
let peers = {}, trafficPrev = {};
let micMuted = false;
let audioCtx, sourceNode, gainNode, cmpNode, hpfNode, lpfNode, destNode, meterNode, monitorNode;
let monitorAudio = null;

// VU
let smoothRmsDb = -120, smoothPeakDb = -120;
const ATTACK = 0.4, RELEASE = 0.06, PEAK_HOLD_MS = 750;
let peakHoldDb = -120, peakHoldUntil = 0;
const MIN_DB = -72, RED_DB = -3;

// ---------- Utils ----------
const log = (...a)=>{ const t=new Date().toLocaleTimeString(); logArea.textContent += `[${t}] ${a.join(' ')}\n`; logArea.scrollTop = logArea.scrollHeight; };
const clearLog = ()=>{ logArea.textContent=''; log('🧹 Log limpiado'); };
const updateVolLabel = ()=> volLabel.textContent = 'x' + Number(volumen.value).toFixed(2);
function showToast(msg){
  toast.textContent = msg; toast.classList.add('show');
  setTimeout(()=> toast.classList.remove('show'), 1500);
}

// ---------- Dispositivos ----------
async function cargarDispositivos(){
  const devs = await navigator.mediaDevices.enumerateDevices();
  micSelect.innerHTML = '';
  devs.filter(d=>d.kind==='audioinput').forEach((d,i)=>{
    const opt = document.createElement('option');
    opt.value = d.deviceId;
    opt.textContent = d.label || `Micrófono ${i+1}`;
    micSelect.appendChild(opt);
  });
}
for(let i=1;i<=24;i++){ const o=document.createElement('option'); o.value=`Sala ${i}`; o.textContent=`Sala ${i}`; salaSelect.appendChild(o); }

// ---------- Grafo ----------
function destroyGraph(){
  [sourceNode,gainNode,cmpNode,hpfNode,lpfNode,meterNode,monitorNode,destNode].forEach(n=>{try{n && n.disconnect()}catch{}});
}
function buildGraph(){
  if(!rawStream) throw new Error('No hay rawStream');
  audioCtx = audioCtx || new AudioContext();
  destroyGraph();

  sourceNode = audioCtx.createMediaStreamSource(rawStream);
  gainNode = audioCtx.createGain(); gainNode.gain.value = Number(volumen.value);

  cmpNode = audioCtx.createDynamicsCompressor();
  cmpNode.threshold.value = Number(cmpThresh.value);
  cmpNode.ratio.value = Number(cmpRatio.value);
  cmpNode.knee.value = Number(cmpKnee.value);
  cmpNode.attack.value = Number(cmpAttack.value);
  cmpNode.release.value = Number(cmpRelease.value);

  hpfNode = audioCtx.createBiquadFilter(); hpfNode.type='highpass'; hpfNode.frequency.value = Number(hpfFreq.value); hpfNode.Q.value=0.707;
  lpfNode = audioCtx.createBiquadFilter(); lpfNode.type='lowpass';  lpfNode.frequency.value = Number(lpfFreq.value); lpfNode.Q.value=0.707;

  meterNode = audioCtx.createAnalyser(); meterNode.fftSize = 1024;
  destNode = audioCtx.createMediaStreamDestination();

  // Monitor
  if (monitorChk.checked) {
    monitorNode = audioCtx.createMediaStreamDestination();
    if (!monitorAudio) { monitorAudio = new Audio(); monitorAudio.autoplay = true; }
    monitorAudio.srcObject = monitorNode.stream;
    monitorAudio.muted = false; monitorAudio.play().catch(()=>{});
  } else {
    monitorNode = null;
    if (monitorAudio) { monitorAudio.pause(); monitorAudio.srcObject = null; }
  }

  // Cadena
  let chain = sourceNode;
  if (hpfOn.checked) { chain.connect(hpfNode); chain = hpfNode; }
  if (lpfOn.checked) { chain.connect(lpfNode); chain = lpfNode; }
  if (cmpOn.checked) { chain.connect(cmpNode); chain = cmpNode; }
  chain.connect(gainNode);
  gainNode.connect(meterNode);
  meterNode.connect(destNode);
  if (monitorNode) meterNode.connect(monitorNode);

  localStream = destNode.stream;

  // reset ballistics
  smoothRmsDb = smoothPeakDb = peakHoldDb = -120; peakHoldUntil = 0;
}
function updateCmpParams(){
  if(!cmpNode) return;
  cmpNode.threshold.value = Number(cmpThresh.value);
  cmpNode.ratio.value = Number(cmpRatio.value);
  cmpNode.knee.value = Number(cmpKnee.value);
  cmpNode.attack.value = Number(cmpAttack.value);
  cmpNode.release.value = Number(cmpRelease.value);
}

// ---------- Aplicar audio ----------
function constraintsFromUI(){
  const id = micSelect.value;
  return {
    audio: {
      deviceId: id ? { exact:id } : undefined,
      autoGainControl: agc.checked,
      echoCancellation: echo.checked,
      noiseSuppression: noise.checked
    }
  };
}
async function applyAudio(){
  const c = constraintsFromUI();
  if (rawStream) rawStream.getTracks().forEach(t=>t.stop());
  rawStream = await navigator.mediaDevices.getUserMedia(c);
  buildGraph();

  // Reemplazo en caliente
  const newTrack = localStream.getAudioTracks()[0];
  let replaced=0;
  for(const peer of Object.values(peers)){
    peer.getSenders().forEach(s=>{
      if(s.track && s.track.kind==='audio'){ s.replaceTrack(newTrack); replaced++; }
    });
  }
  if (replaced>0) log(`♻️ Pista reemplazada en ${replaced} sender(s)`);
  log('✅ Audio aplicado');
  showToast('Audio aplicado');
}

// ---------- WebRTC ----------
function crearPeer(remoteId){
  const peer = new RTCPeerConnection({ iceServers:[{urls:"stun:stun.l.google.com:19302"}] });
  peers[remoteId] = peer; trafficPrev[remoteId] = { bytesSent:0, timestamp:Date.now() };
  if(localStream) localStream.getTracks().forEach(tr=>peer.addTrack(tr, localStream));

  peer.onicecandidate = ev=>{
    if(ev.candidate && ws?.readyState===WebSocket.OPEN){
      ws.send(JSON.stringify({ type:"webrtc-signal", sala: salaSelect.value, payload:{ to:remoteId, signal: ev.candidate }}));
    }
  };
  peer.onconnectionstatechange = ()=>{
    if(["disconnected","failed","closed"].includes(peer.connectionState)){
      log(`❌ Peer desconectado: ${remoteId}`);
      peer.close(); delete peers[remoteId]; delete trafficPrev[remoteId];
    }
  };
  return peer;
}
async function iniciar(){
  try{
    await applyAudio();
    ws = new WebSocket("wss://ws.qsobox.com:4000");
    ws.onopen = ()=>{
      log("📡 WebSocket conectado");
      ws.send(JSON.stringify({ type:"join", sala:salaSelect.value }));
      ws.send(JSON.stringify({ type:"webrtc-user-joined", sala:salaSelect.value }));
      ws.send(JSON.stringify({ type:"soy-emisor" }));
    };
    ws.onmessage = async (ev)=>{
      const data = JSON.parse(ev.data);
      if (data.type==='recuento-receptores'){ usuariosConectados.textContent = `🎧 Receptores: ${data.total}`; return; }
      if (data.type==='webrtc-request-offer'){
        const id = data.from; log("📞 Oferta solicitada por:", id);
        const peer = crearPeer(id);
        const offer = await peer.createOffer(); await peer.setLocalDescription(offer);
        ws.send(JSON.stringify({ type:"webrtc-signal", sala:salaSelect.value, payload:{ to:id, signal: peer.localDescription }}));
        log("📤 Oferta enviada");
      }
      if (data.type==='webrtc-signal'){
        const id = data.from; const peer = peers[id]; if(!peer) return;
        if (data.payload.signal.type==="answer"){
          await peer.setRemoteDescription(new RTCSessionDescription(data.payload.signal));
          log("✅ Answer recibido");
        } else if (data.payload.signal.candidate){
          await peer.addIceCandidate(new RTCIceCandidate(data.payload.signal));
        }
      }
    };
    ws.onclose = ()=>{ log("❌ WebSocket cerrado"); usuariosConectados.textContent = "🎧 Receptores: --"; };

    startBtn.disabled = true; stopBtn.disabled = false; muteBtn.disabled = false; muteBtn.textContent="Silenciar micrófono";
    log("🎙️ Transmisión iniciada");
  }catch(e){ log("⚠️ Error al iniciar:", e.message); alert("Error: "+e.message); }
}
function detener(){
  Object.values(peers).forEach(p=>p.close());
  peers={}; trafficPrev={};
  if (ws) try{ ws.close(); }catch{} ws=null;
  if (localStream) localStream.getTracks().forEach(t=>t.stop());
  if (rawStream) rawStream.getTracks().forEach(t=>t.stop());
  destroyGraph();
  startBtn.disabled=false; stopBtn.disabled=true; muteBtn.disabled=true;
  usuariosConectados.textContent="🎧 Receptores: --"; traficoDiv.textContent="📶 Tráfico total: --";
  log("🛑 Transmisión detenida");
  showToast('Transmisión detenida');
}
function toggleMute(){
  if(!localStream) return;
  const track = localStream.getAudioTracks()[0];
  micMuted = !micMuted; track.enabled = !micMuted;
  muteBtn.textContent = micMuted ? "Activar micrófono" : "Silenciar micrófono";
  log(micMuted ? "🔇 Micrófono silenciado" : "🎙️ Micrófono activado");
}

// ---------- Tráfico ----------
let trafficTotalBytes = 0;
async function actualizarTrafico(){
  let totalRateBytes=0, detalles=[];
  for(const [id,peer] of Object.entries(peers)){
    const stats = await peer.getStats();
    stats.forEach(r=>{
      if(r.type==="outbound-rtp" && !r.isRemote){
        const cur = r.bytesSent||0;
        const prev = trafficPrev[id]?.bytesSent||0;
        const tNow = Date.now();
        const tPrev = trafficPrev[id]?.timestamp||tNow;
        const elapsed = (tNow - tPrev)/1000 || 1;
        const diff = cur - prev;
        const rateKBps = (diff/1024)/elapsed;
        trafficPrev[id] = { bytesSent:cur, timestamp:tNow };
        totalRateBytes += diff; trafficTotalBytes += diff;
        detalles.push(`👤 ${id.slice(0,5)}: ${(cur/1024/1024).toFixed(2)} MB (${rateKBps.toFixed(1)} KB/s)`);
      }
    });
  }
  const totalMB = (trafficTotalBytes/1024/1024).toFixed(2);
  const totalRateKBps = (totalRateBytes/1024/3).toFixed(1);
  traficoDiv.textContent = `📶 Total: ${totalMB} MB (${totalRateKBps} KB/s)\n` + detalles.join('\\n');
}

// ---------- Vúmetro (RMS + peak + hold) ----------
(function meterLoop(){
  const an = meterNode;
  if (an) {
    const buf = new Float32Array(an.fftSize);
    an.getFloatTimeDomainData(buf);

    let sum=0, peak=0, clipped=false;
    for (let i=0;i<buf.length;i++){
      const v = buf[i]; sum += v*v;
      const a = Math.abs(v); if (a>peak) peak=a; if (a>=0.999) clipped=true;
    }
    const rms = Math.sqrt(sum / buf.length);
    const rmsDbInst  = 20*Math.log10(rms || 1e-9);
    const peakDbInst = 20*Math.log10(peak || 1e-9);

    smoothRmsDb  = (rmsDbInst  > smoothRmsDb)  ? smoothRmsDb*(1-ATTACK)  + rmsDbInst*ATTACK  : smoothRmsDb*(1-RELEASE) + rmsDbInst*RELEASE;
    smoothPeakDb = (peakDbInst > smoothPeakDb) ? smoothPeakDb*(1-ATTACK) + peakDbInst*ATTACK : smoothPeakDb*(1-RELEASE)+ peakDbInst*RELEASE;

    const now = performance.now();
    if (smoothPeakDb > peakHoldDb + 0.1) { peakHoldDb = smoothPeakDb; peakHoldUntil = now + PEAK_HOLD_MS; }
    else if (now > peakHoldUntil) { peakHoldDb = peakHoldDb - 0.5; }

    const clamp=(x,min,max)=>Math.max(min,Math.min(max,x));
    const dbToPct = (db)=> clamp((db - MIN_DB) / (0 - MIN_DB) * 100, 0, 100);

    const pct = dbToPct(smoothRmsDb);
    meterEl.style.width = pct + '%';

    const peakPct = dbToPct(peakHoldDb);
    peakHoldEl.style.left = peakPct + '%';

    rmsDbEl.textContent  = (isFinite(smoothRmsDb) ? smoothRmsDb.toFixed(1) : '−∞') + ' dBFS';
    peakDbEl.textContent = (isFinite(smoothPeakDb) ? smoothPeakDb.toFixed(1) : '−∞') + ' dBFS';
    clipDot.classList.toggle('on', clipped || smoothPeakDb >= -0.2);

    if (smoothPeakDb >= RED_DB) {
      meterEl.style.width = Math.max(pct, 98) + '%';
    }
  } else {
    meterEl.style.width = '0%';
  }
  requestAnimationFrame(meterLoop);
})();

// ---------- Guardar / Cargar (3 ranuras) ----------
const CFG_KEY_BASE = 'webrtc_audio_cfg_v3_slot'; // v3 + ranura
function cfgKeyFor(slot){
  const s = Number(cfgSlot?.value || 1);
  return `${CFG_KEY_BASE}${s}`;
}
function readCfg(){
  return {
    micId: micSelect.value,
    agc: agc.checked, echo: echo.checked, noise: noise.checked,
    cmpOn: cmpOn.checked, hpfOn: hpfOn.checked, lpfOn: lpfOn.checked,
    cmpThresh: Number(cmpThresh.value), cmpRatio: Number(cmpRatio.value), cmpKnee: Number(cmpKnee.value),
    cmpAttack: Number(cmpAttack.value), cmpRelease: Number(cmpRelease.value),
    hpfFreq: Number(hpfFreq.value), lpfFreq: Number(lpfFreq.value),
    gain: Number(volumen.value)
  };
}
function writeCfg(c){
  if (!c) return;
  // micId se aplicará al final si existe
  agc.checked=c.agc; echo.checked=c.echo; noise.checked=c.noise;
  cmpOn.checked=c.cmpOn; hpfOn.checked=c.hpfOn; lpfOn.checked=c.lpfOn;
  cmpThresh.value=c.cmpThresh; cmpRatio.value=c.cmpRatio; cmpKnee.value=c.cmpKnee;
  cmpAttack.value=c.cmpAttack; cmpRelease.value=c.cmpRelease;
  hpfFreq.value=c.hpfFreq; lpfFreq.value=c.lpfFreq;
  volumen.value=c.gain; updateVolLabel();
  if (c.micId){
    const opt = [...micSelect.options].find(o=>o.value===c.micId);
    if (opt) micSelect.value = c.micId;
  }
}
saveCfg.onclick = ()=>{
  try{
    const key = cfgKeyFor();
    localStorage.setItem(key, JSON.stringify(readCfg()));
    showToast(`Ajustes guardados (slot ${cfgSlot.value})`);
    log(`💾 Ajustes guardados en ranura ${cfgSlot.value}`);
  }catch(e){
    alert('No se pudieron guardar ajustes (¿bloqueo de almacenamiento?).');
    log('⚠️ Error guardando ajustes: ' + e.message);
  }
};
loadCfg.onclick = async ()=>{
  try{
    const key = cfgKeyFor();
    const s = localStorage.getItem(key);
    if(!s){ showToast(`Ranura ${cfgSlot.value} vacía`); return; }
    const cfg = JSON.parse(s);
    writeCfg(cfg);
    // Reaplica grafo y (si procede) cambia el micrófono
    await applyAudio();
    showToast(`Ajustes cargados (slot ${cfgSlot.value})`);
    log(`📂 Ajustes cargados de la ranura ${cfgSlot.value} y aplicados`);
  }catch(e){
    alert('No se pudieron cargar ajustes.');
    log('⚠️ Error cargando ajustes: ' + e.message);
  }
};

// ---------- Eventos UI ----------
startBtn.onclick = iniciar;
stopBtn.onclick  = detener;
muteBtn.onclick  = toggleMute;
clearLogBtn.onclick = clearLog;

applyAudioBtn.onclick = applyAudio;

applyFxBtn.onclick = ()=> applyAudio(); // reconstruye grafo respetando checks + parámetros
resetFxBtn.onclick = ()=>{
  cmpOn.checked=false; hpfOn.checked=false; lpfOn.checked=false;
  cmpThresh.value=-24; cmpRatio.value=3; cmpKnee.value=30; cmpAttack.value=0.003; cmpRelease.value=0.25;
  hpfFreq.value=90; lpfFreq.value=16000;
  applyAudio();
};

volumen.oninput = ()=>{ updateVolLabel(); if(gainNode) gainNode.gain.value = Number(volumen.value); };

[cmpThresh,cmpRatio,cmpKnee,cmpAttack,cmpRelease].forEach(el=> el.oninput = ()=>updateCmpParams());
[hpfFreq,lpfFreq].forEach(el=> el.oninput = ()=>{ if(hpfNode) hpfNode.frequency.value = Number(hpfFreq.value); if(lpfNode) lpfNode.frequency.value = Number(lpfFreq.value); });
[cmpOn,hpfOn,lpfOn,monitorChk].forEach(el => el.onchange = ()=> applyAudio());

document.getElementById('resetGain').onclick = ()=>{ volumen.value=1; updateVolLabel(); if(gainNode) gainNode.gain.value=1; };

presetDirecto.onclick   = ()=> setPreset({ agc:false, echo:false, noise:false, cmpOn:false, hpfOn:true,  lpfOn:false, cmpThresh:-24, cmpRatio:2, cmpKnee:20, cmpAttack:0.01, cmpRelease:0.3, hpfFreq:90,  lpfFreq:16000, gain:1.00 }, 'Directo');
presetVoz.onclick       = ()=> setPreset({ agc:false, echo:true,  noise:true,  cmpOn:true,  hpfOn:true,  lpfOn:true,  cmpThresh:-28, cmpRatio:3, cmpKnee:30, cmpAttack:0.005,cmpRelease:0.25, hpfFreq:100, lpfFreq:14000, gain:1.20 }, 'Voz limpia');
presetAgc.onclick       = ()=> setPreset({ agc:true,  echo:true,  noise:true,  cmpOn:false, hpfOn:false, lpfOn:false, cmpThresh:-24, cmpRatio:2, cmpKnee:20, cmpAttack:0.01, cmpRelease:0.3, hpfFreq:80,  lpfFreq:16000, gain:1.00 }, 'AGC + Ruido');
presetEntrevista.onclick= ()=> setPreset({ agc:false, echo:true,  noise:true,  cmpOn:true,  hpfOn:true,  lpfOn:true,  cmpThresh:-18, cmpRatio:4, cmpKnee:25, cmpAttack:0.003,cmpRelease:0.2,  hpfFreq:120, lpfFreq:12000, gain:1.10 }, 'Entrevista');
presetRadio.onclick     = ()=> setPreset({ agc:false, echo:false, noise:false, cmpOn:true,  hpfOn:true,  lpfOn:true,  cmpThresh:-20, cmpRatio:6, cmpKnee:35, cmpAttack:0.002,cmpRelease:0.18, hpfFreq:90,  lpfFreq:11000, gain:1.30 }, 'Radio');
presetReset.onclick     = ()=> setPreset({ agc:true,  echo:true,  noise:true,  cmpOn:false, hpfOn:false, lpfOn:false, cmpThresh:-24, cmpRatio:3, cmpKnee:30, cmpAttack:0.003,cmpRelease:0.25, hpfFreq:90,  lpfFreq:16000, gain:1.00 }, 'Reset');

async function setPreset(p, name){
  agc.checked = p.agc; echo.checked = p.echo; noise.checked = p.noise;
  cmpOn.checked = p.cmpOn; hpfOn.checked = p.hpfOn; lpfOn.checked = p.lpfOn;
  cmpThresh.value = p.cmpThresh; cmpRatio.value = p.cmpRatio; cmpKnee.value = p.cmpKnee;
  cmpAttack.value = p.cmpAttack; cmpRelease.value = p.cmpRelease;
  hpfFreq.value = p.hpfFreq; lpfFreq.value = p.lpfFreq;
  volumen.value = p.gain; updateVolLabel();
  try{ await applyAudio(); log(`✅ Preset: ${name}`); showToast('Preset aplicado'); }catch(e){ log("⚠️ Error preset:", e.message); }
}

// ---------- Init ----------
(async ()=>{
  updateVolLabel();
  try{ await navigator.mediaDevices.getUserMedia({ audio:true }); await cargarDispositivos(); }catch{ log('❗ Permiso de micrófono denegado para listar dispositivos'); }
  setInterval(actualizarTrafico, 3000);
  setInterval(()=>{ for(const [id,p] of Object.entries(peers)){ if(p.connectionState!=='connected'){ p.close(); delete peers[id]; delete trafficPrev[id]; } } }, 15000);
})();

// Botones sesión
startBtn.onclick = iniciar;
stopBtn.onclick  = detener;
