// QSO Box server – stores 'code' in qso-pending.json (visible) – at server root
const express = require('express');
const fs = require('fs');
const fsp = require('fs/promises');
const path = require('path');

const app = express();
const ROOT = __dirname;
const PORT = process.env.PORT || 3000;
const WRITE_CODE = process.env.QSO_WRITE_CODE || null; // if set, must match

app.use((req,res,next)=>{
  res.setHeader('Access-Control-Allow-Origin', '*');
  res.setHeader('Access-Control-Allow-Methods', 'GET,PUT,POST,DELETE,OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type, x-qso-code');
  if (req.method === 'OPTIONS') return res.end();
  next();
});

app.use(express.json({ limit: '1mb' }));
app.use(express.static(ROOT, { extensions: ['html'] }));

async function writeJsonSafe(filePath, obj){
  await fsp.mkdir(path.dirname(filePath), { recursive: true });
  const tmp = filePath + '.tmp';
  await fsp.writeFile(tmp, JSON.stringify(obj, null, 2), 'utf8');
  await fsp.rename(tmp, filePath);
}

function getProvidedCode(req){
  return req.header('x-qso-code') || (req.body && String(req.body.code || ''));
}
function isCodeAllowed(req){
  if (!WRITE_CODE) return true;
  return getProvidedCode(req) === WRITE_CODE;
}

async function savePending(body, code){
  const out = {
    pending: true,
    label: 'QSO Pending',
    set: {},
    code: code ? String(code) : '',   // <-- se guarda el código tal cual
    timestamp: Date.now()
  };
  if (body && typeof body === 'object'){
    if (body.set && typeof body.set === 'object'){
      if (typeof body.set.freqHz === 'number') out.set.freqHz = Math.round(body.set.freqHz);
      if (typeof body.set.mode === 'string') out.set.mode = String(body.set.mode).toUpperCase();
    }
  }
  await writeJsonSafe(path.join(ROOT, 'qso-pending.json'), out);
  return out;
}

app.put('/api/qso/pending', async (req, res)=>{
  try{
    if (!isCodeAllowed(req)) return res.status(403).json({ ok:false, error:'forbidden' });
    const out = await savePending(req.body, getProvidedCode(req));
    res.json({ ok:true, out });
  }catch(err){ console.error('save pending failed', err); res.status(500).json({ ok:false, error:'write_failed' }); }
});
app.post('/api/qso/pending', async (req, res)=>{
  try{
    if (!isCodeAllowed(req)) return res.status(403).json({ ok:false, error:'forbidden' });
    const out = await savePending(req.body, getProvidedCode(req));
    res.json({ ok:true, out });
  }catch(err){ console.error('save pending failed', err); res.status(500).json({ ok:false, error:'write_failed' }); }
});

app.get('/qso-pending.json', async (req, res)=>{
  try{
    const fp = path.join(ROOT, 'qso-pending.json');
    if (!fs.existsSync(fp)) return res.json({});
    res.sendFile(fp);
  }catch{ res.json({}); }
});

app.listen(PORT, ()=> console.log('QSO server at http://localhost:'+PORT+' (WRITE_CODE '+(WRITE_CODE?'ON':'OFF')+')'));
